/*
 * Copyright (c) 2006 Oracle.  All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */
#include <linux/kernel.h>
#include <linux/sysctl.h>
#include <linux/proc_fs.h>

#include "rds.h"

static struct ctl_table_header *rds_sysctl_reg_table;

static unsigned long rds_sysctl_reconnect_min = 1;
static unsigned long rds_sysctl_reconnect_max = ~0UL;

unsigned long rds_sysctl_reconnect_min_jiffies;
unsigned long rds_sysctl_reconnect_max_jiffies = HZ;

/* 
 * These can change over time until they're official.  Until that time we'll
 * give apps a way to figure out what the values are in a given machine.
 */
static int rds_sysctl_pf_rds = PF_RDS;
static int rds_sysctl_sol_rds = SOL_RDS;

static ctl_table rds_sysctl_rds_table[] = {
	{
		.ctl_name       = 4,
		.procname       = "reconnect_min_delay_ms",
		.data		= &rds_sysctl_reconnect_min_jiffies,
		.maxlen         = sizeof(unsigned long),
		.mode           = 0644,
		.proc_handler   = &proc_doulongvec_ms_jiffies_minmax,
		.extra1		= &rds_sysctl_reconnect_min,
		.extra2		= &rds_sysctl_reconnect_max_jiffies,
	},
	{
		.ctl_name       = 5,
		.procname       = "reconnect_max_delay_ms",
		.data		= &rds_sysctl_reconnect_max_jiffies,
		.maxlen         = sizeof(unsigned long),
		.mode           = 0644,
		.proc_handler   = &proc_doulongvec_ms_jiffies_minmax,
		.extra1		= &rds_sysctl_reconnect_min_jiffies,
		.extra2		= &rds_sysctl_reconnect_max,
	},
	{
		.ctl_name       = 6,
		.procname       = "pf_rds",
		.data		= &rds_sysctl_pf_rds,
		.maxlen         = sizeof(int),
		.mode           = 0444,
		.proc_handler   = &proc_dointvec,
	},
	{
		.ctl_name       = 7,
		.procname       = "sol_rds",
		.data		= &rds_sysctl_sol_rds,
		.maxlen         = sizeof(int),
		.mode           = 0444,
		.proc_handler   = &proc_dointvec,
	},
	/* 100+ are reserved for transport subdirs */
	{ .ctl_name = 0}
};

static ctl_table rds_sysctl_net_table[] = {
	{
		.ctl_name	= SYSCTL_NET_RDS,
		.procname	= "rds",
		.data		= NULL,
		.maxlen		= 0,
		.mode		= 0555,
		.child		= rds_sysctl_rds_table
	},
	{ .ctl_name = 0}
};

static ctl_table rds_sysctl_root_table[] = {
	{
		.ctl_name	= CTL_NET,
		.procname	= "net",
		.data		= NULL,
		.maxlen		= 0,
		.mode		= 0555,
		.child		= rds_sysctl_net_table
	},
	{ .ctl_name = 0 }
};

void __exit rds_sysctl_exit(void)
{
	if (rds_sysctl_reg_table)
		unregister_sysctl_table(rds_sysctl_reg_table);
}

int __init rds_sysctl_init(void)
{
	rds_sysctl_reconnect_min = msecs_to_jiffies(1);
	rds_sysctl_reconnect_min_jiffies = rds_sysctl_reconnect_min;

	rds_sysctl_reg_table = register_sysctl_table(rds_sysctl_root_table, 0);
	if (rds_sysctl_reg_table == NULL)
		return -ENOMEM;
	return 0;
}
